
' This plug-in is used with the Plug-in (.NET) Sample Application on DevNet and is intended to provide
' coding examples for various plug-in usage scenarios, using different Logi plug-in elements
'
' *** Plug-ins are cached, so you will need to stop/restart your web server before making each new build ***
'
' Configurations for this sample:
' In Visual Studio: Project -> Properties -> Application tab -> Root namespace = LogiXML.SamplePlugin
' In Logi Studio: Class Type Name attributes for Plug-in elements should be set to "LogiXML.SamplePlugin.Plugin"
'
' Note this very useful method: rdPlugin.rdServerObjects.AddDebugMessage("Plug-in Message", "Your message") which
' adds a message to the Debug Trace Page and can help you understand what's going on in a plug-in
'
' For detailed information about creating and using plug-ins, see these DevNet documents:
'
' Introducing Logi Plug-ins - http://devnet.logixml.com/rdPage.aspx?rdReport=Article&dnDocID=2189
' Creating a .NET Plug-in -   http://devnet.logixml.com/rdPage.aspx?rdReport=Article&dnDocID=2190
' Creating a Java Plug-in -   http://devnet.logixml.com/rdPage.aspx?rdReport=Article&dnDocID=2227

' Visual Studio 2012 - March 2016 - L. Hausman
'
'

Imports System.Configuration
Imports System.Xml

Public Class Plugin

    ' demonstrates setting an element's attribute to a new value;
    ' called by the Plugin Call element in the _Settings definition

    Public Sub SetApplicationCaption(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' load the _Settings definition XML
        Dim xmlSettings As New XmlDocument()
        xmlSettings.LoadXml(rdObjects.CurrentDefinition)

        ' select the Application element and set its Caption attribute
        Dim eleApp As XmlElement = xmlSettings.SelectSingleNode("//Setting/Application")
        eleApp.SetAttribute("Caption", "Greetings from the Sample Plugin!  Time: " & Now.ToString)

        ' replace the _Settings definition with the altered XML
        rdObjects.CurrentDefinition = xmlSettings.OuterXml

    End Sub

    ' demonstrates using a plugin to insert rows of data into a datalayer;
    ' called by Data Layer Plugin Call element in the Data_Layer_Plugin_Call definition

    Public Sub GetProductData(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' access the data from the datalayer - in this case, the data is being 
        ' passed as an XML document ("Pass Data As" attribute)
        Dim xmlData As New XmlDocument()
        xmlData = rdObjects.CurrentData

        ' get name of the parent data table, which was passed as a plugin parameter
        Dim sParentDataTableID As String = rdObjects.PluginParameters("ParentDataTableID")

        ' insert multiple records
        Dim eleRow As XmlElement
        eleRow = xmlData.DocumentElement.AppendChild(xmlData.CreateElement(sParentDataTableID))
        eleRow.SetAttribute("ProductName", "Milk")
        eleRow = xmlData.DocumentElement.AppendChild(xmlData.CreateElement(sParentDataTableID))
        eleRow.SetAttribute("ProductName", "Bread")
        eleRow = xmlData.DocumentElement.AppendChild(xmlData.CreateElement(sParentDataTableID))
        eleRow.SetAttribute("ProductName", "Beer")

        ' we do not need to assign xmlData back to CurrentData! Why? Because it  
        ' was passed as an XML Document to us from the definition
        ' XX rdObjects.CurrentData = xmlData XX

    End Sub

    ' demonstrates using a plugin to insert a new column into a datalayer;
    ' called by the Data Layer Plugin Call element in the Data_Layer_Plugin_Call definition

    Public Sub AddNewColumn(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' access the data from the datalayer - in this case, the data is being 
        ' passed as an XML document ("Pass Data As" attribute)
        Dim xmlData As New XmlDocument()
        xmlData = rdObjects.CurrentData

        ' this code is not necessary for this sample but is included to show how to handle data being 
        ' passed either as an XML document OR an XML file (the two "Pass Data As" attribute options)
        '
        'Dim sReturnFile As String = Nothing
        'If (rdObjects.CurrentData IsNot Nothing) Then
        '    xmlData = rdObjects.CurrentData
        'ElseIf (rdObjects.CurrentDataFile IsNot Nothing) AndAlso (IO.File.Exists(rdObjects.CurrentDataFile)) Then
        '    xmlData = New XmlDocument()
        '    xmlData.Load(rdObjects.CurrentDataFile)
        '    sReturnFile = rdObjects.ReturnedDataFile
        'Else
        '    Throw New Exception("No XML data was passed to the AddNewColumn plug-in method.")
        'End If

        ' get name of the new column, which was passed as a plugin parameter
        Dim sColumnName As String = rdObjects.PluginParameters("NewColumnName")

        ' iterate each data row, inserting the new column and its value
        Dim eleRow As XmlElement
        For Each eleRow In xmlData.SelectNodes("/rdData/*")
            Select Case eleRow.GetAttribute("ProductName")
                Case "Milk"
                    eleRow.SetAttribute(sColumnName, "3.25")
                Case "Bread"
                    eleRow.SetAttribute(sColumnName, "3.75")
                Case "Beer"
                    eleRow.SetAttribute(sColumnName, "4.50")
            End Select
        Next

        ' if the current data had been passed to us as a file, we  
        ' would need to save the updated file back to the caller
        'If (sReturnFile IsNot Nothing) Then
        'xmlData.Save(sReturnFile)
        'End If

    End Sub

    ' demonstrates using a plugin to retrieve data from sources not directly supported by other datalayer elements;
    ' called by the DataLayer.Plugin element in the DataLayer.Plugin_Call definition

    ' this example uses an MS Access data file as a datasource, which requires that you have the Jet driver installed,
    ' however, the Jet driver is not available for 64-bit systems and is deprecated in later versions of Office, so it
    ' may not be available to you
    ' it's used here simply as a matter of convenience; any datasource could be substituted for it

    Public Sub GetMSAccessData(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' access the data from the datalayer - in this case, the data is being 
        ' passed as an XML document ("Pass Data As" attribute)
        Dim xmlData As New XmlDocument()
        xmlData = rdObjects.CurrentData

        ' define OLEDB objects
        Dim connOleDB As OleDb.OleDbConnection
        Dim daOleDB As OleDb.OleDbDataAdapter
        Dim dset As New DataSet

        ' get the application root folder path, by reading a token
        Dim AppRootFolder As String = rdObjects.ReplaceTokens("@Function.AppPhysicalPath~")

        ' get name of the parent data table, which was passed as a plugin parameter
        Dim sParentDataTableID As String = rdObjects.PluginParameters("ParentDataTableID")

        ' open OLEDB connection to MS Access data file
        connOleDB = New OleDb.OleDbConnection("Provider=Microsoft.Jet.OLEDB.4.0;Data Source=" & AppRootFolder & "\Data\NWIND.MDB")
        daOleDB = New OleDb.OleDbDataAdapter("SELECT EmployeeID, LastName, FirstName, Title FROM Employees", connOleDB)
        daOleDB.Fill(dset, "Employees")

        ' read the data from the file and insert it into the datalayer
        Dim x As Integer
        Dim newNode As XmlNode
        Dim newAtt As XmlAttribute
        Dim dataRoot As XmlElement = xmlData.SelectSingleNode("//rdData")

        For x = 0 To dset.Tables("Employees").Rows.Count - 1

            ' create the XML for a single datalayer row
            newNode = xmlData.CreateElement(sParentDataTableID)

            newAtt = xmlData.CreateAttribute("EmployeeID")
            newAtt.Value = dset.Tables("Employees").Rows(x).Item(0)
            newNode.Attributes.Append(newAtt)

            newAtt = xmlData.CreateAttribute("LastName")
            newAtt.Value = dset.Tables("Employees").Rows(x).Item(1)
            newNode.Attributes.Append(newAtt)

            newAtt = xmlData.CreateAttribute("FirstName")
            newAtt.Value = dset.Tables("Employees").Rows(x).Item(2)
            newNode.Attributes.Append(newAtt)

            newAtt = xmlData.CreateAttribute("Title")
            newAtt.Value = dset.Tables("Employees").Rows(x).Item(3)
            newNode.Attributes.Append(newAtt)

            ' add the new row to the dalayer
            dataRoot.AppendChild(newNode)

        Next

        ' close the connection
        connOleDB.Close()

    End Sub

    ' demonstrates the use of a plugin to modify the XML definition for a super-element;
    ' called by the Generated Element Plugin Call element in the Generated_Element_Plugin_Call definition

    Public Sub SetAGAttributes(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' load the report definition XML
        Dim xmlDef As New XmlDocument()
        xmlDef.LoadXml(rdObjects.CurrentDefinition)

        ' when using the Generated Element Plugin Call element, rdServerObjects.CurrentDefinition 
        ' contains only the XML for the super-element, so the XML root node IS the super-element, not the report 
        Dim xmlAG As XmlElement = xmlDef.SelectSingleNode("//AnalysisGrid")

        ' create and append the new attribute, which hides the Export buttons
        Dim newAtt As XmlAttribute
        newAtt = xmlDef.CreateAttribute("HideAgExport")
        newAtt.Value = "True"
        xmlAG.Attributes.Append(newAtt)

        ' replace the report definition with the altered XML
        rdObjects.CurrentDefinition = xmlDef.OuterXml

    End Sub

    ' demonstrates the use of a plugin to modify the XML definition for a dashboard panel;
    ' called by the Panel Plugin Call element in the Panel_Plugin_Call definition

    Public Sub SetDashPanel(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' load the report definition XML
        Dim xmlDef As New XmlDocument()
        xmlDef.LoadXml(rdObjects.CurrentDefinition)

        ' get plugin parameter to determine which action to take
        Dim PanelAction As Integer = rdObjects.PluginParameters("PanelAction")

        If PanelAction = 1 Then

            ' change the caption of an existing Label within panel to date and time
            Dim eleLabel As XmlElement = xmlDef.DocumentElement.SelectSingleNode(".//Label")
            If Not IsNothing(eleLabel) Then
                eleLabel.SetAttribute("Caption", "This Label Caption value set by Panel Plugin Call: " + DateTime.Now.ToString())
            End If

        ElseIf PanelAction = 2 Then

            ' add a new label element and set two of its attributes 
            Dim elePanel As XmlElement = xmlDef.DocumentElement.SelectSingleNode("//Division")
            Dim eleNewLabel As XmlElement = xmlDef.CreateElement("Label")
            eleNewLabel.SetAttribute("Caption", "This Label element added by Panel Plugin Call: " + DateTime.Now.ToString())
            eleNewLabel.SetAttribute("Class", "font11pt")
            elePanel.AppendChild(eleNewLabel)
            
            ' add a new line element
            Dim eleNewLine As XmlElement = xmlDef.CreateElement("LineBreak")
            eleNewLine.SetAttribute("LineCount", "8")
            elePanel.AppendChild(eleNewLine)

        End If

        ' replace the report definition with the altered XML
        rdObjects.CurrentDefinition = xmlDef.OuterXml

    End Sub

    ' demonstrates the use of a plugin to modify the XML data when the FinishData event occurs;
    ' called by the Plugin Call element in the Plugin_Call_FinishData definition

    Public Sub DoFinishData(ByRef rdObjects As rdPlugin.rdServerObjects)

        Dim sCurrFileName As String = rdObjects.CurrentDataFile
        Dim sReturnFileName As String = rdObjects.ReturnedDataFile

        ' access the data from the datalayers - the data is being passed as an XML file
        Dim xmlData As New XmlDocument()
        xmlData.Load(sCurrFileName)

        ' get plugin parameter to determine whether to take action
        Dim PluginAction As Integer = rdObjects.PluginParameters("PluginAction")

        If PluginAction = 1 Then

            ' iterate the rows for table dtEmployees and change case of value in FirstName column
            For Each dRow As XmlElement In xmlData.DocumentElement.SelectNodes("//dtEmployees")
                If dRow.HasAttribute("FirstName") Then
                    dRow.Attributes("FirstName").Value = dRow.Attributes("FirstName").Value.ToUpper()
                End If
            Next

            ' iterate the rows for table dtCustomers and change case of value in CustomerID column
            For Each dRow As XmlElement In xmlData.DocumentElement.SelectNodes("//dtCustomers")
                If dRow.HasAttribute("CustomerID") Then
                    dRow.Attributes("CustomerID").Value = dRow.Attributes("CustomerID").Value.ToLower()
                End If
            Next

        End If

        ' save the altered data
        xmlData.Save(sReturnFileName)

    End Sub

    ' demonstrates the use of a plugin to modify the report HTML when the FinishHTML event occurs;
    ' called by the Plugin Call element in the Plugin_Call_FinishHTML definition

    Public Sub HighlightText(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' a little different approach for variety: use "With" style of referencing rdObjects
        With rdObjects

            ' get plug-in parameters passed from definition
            Dim sElementID As String = .PluginParameters("ElementID")
            Dim sFindInputID As String = .PluginParameters("FindInputTextID")   ' input.text element
            Dim sStyle As String = .PluginParameters("Style")

            ' get search term entered into input.text element
            Dim sFindValue As String = .Request.Form(sFindInputID)
            If IsNothing(sFindValue) Then sFindValue = .Request(sFindInputID)

            ' if search term was entered, search finished HTML 
            If Not IsNothing(sFindValue) AndAlso sFindValue.Length <> 0 Then

                ' load the ResponsesHtml string into an XML Document
                Dim xmlHtml As New XmlDocument()
                xmlHtml.LoadXml(.ResponseHtml)

                ' find all SPAN elements under the element with the specified Element ID
                Dim nlSpans As XmlNodeList = xmlHtml.SelectNodes("//*[@id='" & sElementID & "']//SPAN")
                Dim eleSpanText As XmlElement

                ' iterate spans and split the SPAN into three SPANs to highlight the first match
                For Each eleSpanText In nlSpans
                    Call subSplitHighlightSpan(xmlHtml, eleSpanText, sFindValue, sStyle)
                Next

                ' save the altered HTML
                .ResponseHtml = xmlHtml.OuterXml

            End If

        End With

    End Sub

    ' sub-routine called by HighlightText
    Private Sub subSplitHighlightSpan(ByVal xmlHtml As XmlDocument, ByVal eleSpanText As XmlElement, ByVal sFindValue As String, ByVal sStyle As String)

        Dim nFindLocation As Integer = eleSpanText.InnerText.ToLower.IndexOf(sFindValue.ToLower)

        If nFindLocation <> -1 Then

            ' search string was found - break the SPAN into three SPANs
            Dim sOriginal As String = eleSpanText.InnerText

            ' set the first SPAN element
            eleSpanText.InnerText = sOriginal.Substring(0, nFindLocation)

            ' create the highlighted SPAN element
            Dim eleNewSpan As XmlElement = eleSpanText.ParentNode.InsertAfter(xmlHtml.CreateElement("SPAN"), eleSpanText)
            eleNewSpan.SetAttribute("style", sStyle)
            eleNewSpan.InnerText = sOriginal.Substring(nFindLocation, sFindValue.Length)

            ' create the final SPAN element
            eleNewSpan = eleSpanText.ParentNode.InsertAfter(xmlHtml.CreateElement("SPAN"), eleNewSpan)
            eleNewSpan.InnerText = sOriginal.Substring(nFindLocation + sFindValue.Length)

            ' recursively call again with this last SPAN as there may be more matches
            Call subSplitHighlightSpan(xmlHtml, eleNewSpan, sFindValue, sStyle)

        End If

    End Sub

    ' demonstrates the use of a plugin to modify the report HTML when the LoadDefinition event occurs,
    ' and also demonstrates reading text from the application's web.config file
    ' called by the Plugin Call element in the Plugin_Call_LoadDef definition

    Public Sub SetTheme(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' load the current definition XML
        Dim xmlDef As New XmlDocument()
        xmlDef.LoadXml(rdObjects.CurrentDefinition)

        ' read the application's web.config file and retrieve theme choice 
        Dim sThemeName As String = ConfigurationManager.AppSettings("ThemeChoice")

        ' select the style element, using XPATH @ID= syntax, and set its theme attribute
        Dim eleStyle As XmlElement = xmlDef.SelectSingleNode("//*[@ID='SpecialStyle']")
        If eleStyle IsNot Nothing Then
            eleStyle.SetAttribute("Theme", sThemeName)
        End If

        ' replace the current definition with the altered XML
        rdObjects.CurrentDefinition = xmlDef.OuterXml

    End Sub

    ' demonstrates the use of a plugin called by a process task 
    ' called by the Procedure.PluginCall element in the SamplePluginTasks definition

    Public Function GetVirtualMemorySize(ByRef rdObjects As rdPlugin.rdServerObjects) As String

        ' query the OS for the amount of available virtual memory
        Dim currProc As System.Diagnostics.Process = Diagnostics.Process.GetCurrentProcess

        Return currProc.VirtualMemorySize64

    End Function

    ' demonstrates the use of a variety of plugin methods for writing to the Debug page, 
    ' resolving tokens, getting plug-in paramters, getting session and request vars, etc. 
    ' called by the Plugin Call element in the Plugin_Misc_Methods definition

    Public Sub DoMiscMethods(ByRef rdObjects As rdPlugin.rdServerObjects)

        ' resolve tokens and write them to the Debug Trace page
        Dim sSomeTokens As String = "Resolved tokens: @Function.Date~  @Request.CompanyName~  @Session.CompanyCityState~"
        Dim sResolvedTokens As String = rdObjects.ReplaceTokens(sSomeTokens)
        rdObjects.AddDebugMessage("Plug-in Message 1", "Resolve Tokens", "View tokens", sResolvedTokens)
        ' another way to do it
        rdObjects.AddDebugMessage("Plug-in Message 2", "Resolve Token", rdObjects.ReplaceTokens("@Date.Today~"))

        ' get plugin parameter passed from report def and write to Debug page
        Dim sPluginParam As String = rdObjects.PluginParameters("MyParameter")   ' case-sensitive!
        rdObjects.AddDebugMessage("Plug-in Message 3", "Parameter", sPluginParam)

        ' get the _Settings definition
        Dim xmlSettings As XmlDocument = rdObjects.SettingsDefinition
        rdObjects.AddDebugMessage("Plug-in Message 4", "_Settings Def", "View settings", xmlSettings)

        ' get request and session variables
        rdObjects.AddDebugMessage("Plug-in Message 5", "Request Var", "rdReport=" & rdObjects.Request("rdReport"))
        rdObjects.AddDebugMessage("Plug-in Message 6", "Session Var", "CompanyCityState=" & rdObjects.Session("CompanyCityState"))

    End Sub


End Class
